#!/usr/bin/python2.3
# -*- coding: UTF-8 -*-

#
# This file is distributed under the terms of GNU GPL v2 or later, at your choice
# Copyright: Eddy Petrişor <eddy.petrisor at gmail.com>
#

# Revision ID : $Id: SelectorTip.py 33 2005-09-25 22:21:04Z eddy $

"""Clasa principală a aplicaţiei OrtoSelector

Citeşte cuvinte dintr-un fişier de intrare şi
permite categorisirea lor în diverse grupe:
comun, nume, englezism, abreviere, neologism, incorect.

Fişierul de intrare trebuie să aibă un cuvânt pe linie.
Sunt omise liniile goale şi tot ceea ce urmează după „#” sau „;”"""


import os
import gtk
import gtk.glade
import re

errmsg = os.sys.stderr.write
dbgmsg = os.sys.stderr.write

# import la modulele specifice aplicaţiei
from CTipCuv import CTipCuv
from CIOHandler import CIOHandler


class MainWindow:
	"""Implementează funcţionalitatea interfeţei grafice
	şi legătura cu clasele folosite
	
	OrtoSelector - clasa de bază a aplicaţiei
	Foloseşte OrtoMain.galde pentru a instanţia interfaţa"""
	def __init__ ( self ):
		
		self.__appdebug__ = False
		self.init_debug ()
		
		self.VIOHandler = CIOHandler()

		# create widget tree ...
		gtk_interface_file = 'OrtoMain.glade'
		self.xml = gtk.glade.XML(gtk_interface_file)

		# avem nevoie şi de un obiect de manipulat fereastra
		self.win_SelectorTipCuv = self.xml.get_widget ( 'win_SelectorTipCuv' )
		self.xml.signal_autoconnect( self )

		# cuvântul curent trebuie afişat şi manevrat cumva
		self.edit_Cuvant = self.xml.get_widget ( 'edit_Cuvant' ) 
		
		self.lista_tipuri = [ 'comun', \
			'neologism', \
			'nume', \
			'abreviere', \
			'englezism', \
			'incorect' ]
		self.lista_butoane = [ 'radio_TipCuvComun', \
			'radio_TipCuvNeologism', \
			'radio_TipCuvNume', \
			'radio_TipCuvAbreviere', \
			'radio_TipCuvEnglezism', \
			'radio_TipCuvIncorect' ]
		# apelul ăsta trebuie să apară după listele de mai sus
		# pentru că presupune existenţa lor
		self.prepare_next_word ()
		
	def init_debug ( self ):
		"""Importă atributul __appdebug__ în obiectul curent"""

		try:
			self.__appdebug__ = __appdebug__
		except NameError:
			self.__appdebug__ = False
			pass

	def __deinit__ ( self ):
		"""Toate acţinile de curăţare, îniante de terminare"""
		self.VIOHandler.__deinit__()

	# câteva metode ale clasei importate dinamic
	
	def gtk_main_quit( self ):
		"""Ieşire din aplicaţie / finalizare buclă gtk"""
		gtk.main_quit()

	
	def on_radio_tipcuv_clicked ( self, widget ):
		"""Selectează şi memorează tipul de cuvânt pe care s-a dat clic"""
		index = self.lista_butoane.index ( widget.get_name() )
		VTipCuv.mem_tipcuv ( self.lista_tipuri [ index ] )
		if self.__appdebug__:
			dbgmsg ( "index = %d\tnume = %s\ttip = %s\n" % \
				(index, \
				self.lista_tipuri [ index ], \
				widget.get_name()) )
		

	"""Butoanele cu acţiunile lor"""
	def on_but_aplica_clicked( self, _ ):
		"""Salvează în fişier cuvântul şi tipul său, după un „#”
		Scopul #-ului e să se poată genera uşor din consolă
		  grupuri de cuvinte"""

		cuvant_curent = VTipCuv.ret_cuv ()
		if ( cuvant_curent != '' ):
			if ( self.__appdebug__ ):
				dbgmsg ( "S-a aplicat tipul %s cuvântului %s\n" % \
					( VTipCuv.ret_tipcuv (), cuvant_curent ) )

			self.VIOHandler.write ( "%s\t#%s\n" % \
				( cuvant_curent, VTipCuv.ret_tipcuv () ))
		
		# trebuie să trecem la următorul cuvânt
		self.prepare_next_word ()


	def on_but_inainte_clicked ( self, _ ):
		"""Sare la cuvântul următor şi nu salvează nimic pentru cel curent"""
		if self.__appdebug__ :
			dbgmsg ( 'Înainte\n' )
			
		self.prepare_next_word ()

	def on_but_inapoi_clicked ( self, _ ):
		"""Sare la cuvântul anterior şi nu salvează nimic pentru cel curent"""
		# oare e nevoie de chestia asta? are ceva utilitate?
		if self.__appdebug__ :
			dbgmsg ( 'Înapoi\n' )
		pass

	def on_but_iesire_clicked( self, _ ):
		"""Acţiune executată atunci când se apasă ieşire"""
		gtk.main_quit ()

	"""Funcţii auxiliare"""
	def prepare_next_word ( self ):
		"""Citeşte următorul cuvânt de la intrare
		şi îl afişează în interfaţă"""
		
		
		linie = self.VIOHandler.get_next_line ()
		
		# bucata de mai jos ar trebui să fie în CTipCuv
		# şi să fie înlocuită cu VTipCuv.extrage_si_mem(linie)
		
		# se taie eventualele comentarii şi caractere albe
		regex = '^(.*?)([#;](.*))?$'
		cuvant_real = re.search ( regex, linie ).group(1).strip()
		# cuvântul citit trebuie să fie valid
		while self.VIOHandler.FisierInNeterminat and cuvant_real == '' :
			linie = self.VIOHandler.get_next_line ()
			cuvant_real = re.search ( regex, linie ).group(1).strip()
		if self.__appdebug__:
			dbgmsg ("<<%s>> extras din <<%s>>\n" % (cuvant_real, linie))
		
		VTipCuv.mem_cuv ( cuvant_real )
		self.edit_Cuvant.set_text ( cuvant_real )
		
		# în cazul în care s-a ajuns la ultima înregistrare
		if cuvant_real != '' :
			self.heuristic_update ()
		else:
			buton_radio = self.xml.get_widget ( self.lista_butoane [ 0 ] )
			buton_radio.set_active(True)
		
		
	def heuristic_update ( self ):
		"""Ghiceşte tipul cuvântului şi selectează
		  butonul radio corespunzător"""
		
		# ghiceşte tipul cuvântului
		VTipCuv.aproximeaza_tipcuv ()
		
		# ce s-a ghicit?
		try:
			indextip = self.lista_tipuri.index ( VTipCuv.ret_tipcuv() )
		except AttributeError:
			# dacă apare această excepţie, atunci 
			#  - fie lista de tipuri nu mai e actuală
			#  - fie în CTipCuv.py e o eroare de ortografie
			errmsg ( 'Tipul aproximat %s nu există în lista de tipuri\n' % \
				VTipCuv.ret_tipcuv () )
			indextip = 0
		
		# în mod normal nu ar trebui să apară probleme aici
		# dacă apar, atunci fişierul glade a fost modificat
		#
		# selectăm butonul radio în conformitate cu tipul estimat
		#
		buton_radio = self.xml.get_widget ( self.lista_butoane [ indextip ] )
		buton_radio.set_active(True)
		
		if self.__appdebug__:
			dbgmsg ( 'S-a selectat %s cu indexul %d şi numele %s\n' % \
				 ( buton_radio, \
				  indextip, \
				  self.lista_butoane [ indextip ] ) )
		
		
##############################################
# begin code

__appdebug__ = False


VTipCuv = CTipCuv()
VMainWin = MainWindow ()

gtk.main()

VMainWin.__deinit__()
VTipCuv.__deinit__()

